%% Comparing Methods of Spectral Analysis
% Martin H. Trauth 2 July 2021
%
% We first clear the workspace and the command window. We then close all
% figure windows.
clear, clc, close all
ds = get(0,'ScreenSize');

%%
% We then load the synthetic data from the file series3.txt. The data
% series contains three main periodicities of 100, 40 and 20 kyrs and
% additive Gaussian noise.
% 
% The amplitudes, however, change through time and the example can
% therefore be used to demonstrate the use of methods to calculate an
% evolutionary power spectrum.
%
% In our example the 40 kyr cycle appears only after ca. 450 kyrs, whereas
% the 100 and 20 kyr cycles are present throughout the time series. We then
% plot the time series.
data = load('series3.txt');

f1 = figure('Position',[50 ds(4)-500 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7])
line(data(:,1),data(:,2),...
    'LineWidth',1)
xlabel('Age (kyr)')
ylabel('Proxy (unit)')

print(f1,'-dpng','-r300','figure_1_timeseries.png')

%%
% The data are irregular spaced. We can calculate the median difference of
% subsequent data points, display the differences through time and in a
% histogram.
median(diff(data(:,1)))

f2 = figure('Position',[50 ds(4)-600 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7])
line(data(1:end-1,1),diff(data(:,1)))
xlabel('Age (kyr)')
ylabel('Delta Age (kyr)')

f3 = figure('Position',[50 ds(4)-700 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7])
histogram(diff(data(:,1)),20)
xlabel('Age Difference (kyr)')
ylabel('Counts')

%%
% Since the data are unevenly-spaced we use the Lomb-Scargle (LS) Method to
% compute the power spectrum. We first define the false alarm probability.
% We then use the function plomb from the Signal Processing Toolbox of
% MATLAB to calculate the LS power spectrum of the full time series.
Pfa = [50 10 1 0.01]/100;
Pd = 1-Pfa;

[pxx,f,pth] = plomb(data(:,2),data(:,1),...
    'normalized','Pd',Pd);

f4 = figure('Position',[50 ds(4)-800 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
    'XLim',[0 0.15],...
    'YLim',[0 40]); hold on
line(f,pxx)
line(f,pth*ones(size(f')))
line([1/20 1/20],...
    [0 max(pxx(:))],...
    'Color','k')
line([1/40 1/40],...
    [0 max(pxx(:))],...
    'Color','k')
line([1/100 1/100],...
    [0 max(pxx(:))],...
    'Color','k')
text(1/20,max(pxx(:))+0.1*max(pxx(:)),...
    'Prec','HorizontalAlignment','center')
text(1/40,max(pxx(:))+0.1*max(pxx(:)),...
    'Obl','HorizontalAlignment','center')
text(1/100,max(pxx(:))+0.1*max(pxx(:)),...
    'Ecc','HorizontalAlignment','center')
text(0.151*[1 1 1 1],pth-.5, ...
    [repmat('P_{fa} = ',[4 1]) num2str(Pfa')])
xlabel('Frequency (1/kyr)')
ylabel('Power')
title('Lomb-Scargle Power Spectrum')
hold off

print(f4,'-dpng','-r300','figure_5c_lombscargle.png')

%%
% We then calculate the LS power spectrum of a sliding window of length w.
% After defining the window size, we pre-allocate memory for the output of
% plomb and then calculate the LS spectra of the sliding window. We then
% interpolate the resulting spectra to the same frequency axis to display
% the evoluationary LS spectrum in a pseudocolor plot.
clear pxx f pth
w = 50;

pxxevol = zeros(2*w+2,length(data));
fevol   = zeros(2*w+2,length(data));
pthevol = zeros(4,length(data));

for i = w/2+1 : length(data)-w/2 
    clear pxxi fi pthi
    [pxxevol(:,i),fevol(:,i),...
        pthevol(:,i)] = ...
        plomb(data(i-w/2:i+w/2,2),...
        data(i-w/2:i+w/2,1),...
        'normalized','Pd',Pd);
end

fint = 0:max(fevol(:))/...
    length(fevol):max(fevol(:));
fint = fint';

pxxint = zeros(length(fint),length(data));

for i = w/2+1 : length(data)-w/2
   pxxint(:,i) = interp1(fevol(:,i),...
       pxxevol(:,i),fint,'linear');
end

[T,F] = meshgrid(data(:,1),fint);
PXX = pxxint;

f5 = figure('Position',[50 ds(4)-900 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
    'YLim',[0 0.15],...
    'ZLim',[0 max(PXX(:))]), hold on
surf(T,F,PXX), shading interp
view(0,90)
plot3([min(data(:,1)) max(data(:,1))],...
    [1/20 1/20],...
    [max(PXX(:)) max(PXX(:))],'w')
plot3([min(data(:,1)) max(data(:,1))],...
    [1/40 1/40],...
    [max(PXX(:)) max(PXX(:))],'w')
plot3([min(data(:,1)) max(data(:,1))],...
    [1/100 1/100],...
    [max(PXX(:)) max(PXX(:))],'w')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/20,'Prec')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/40,'Obl')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/100,'Ecc')
xlabel('Time (kyr)')
ylabel('Frequency (1/kyr)')
titlestr = ...
    ['Evolutionary LS Power Spectrum, w = ',...
    num2str(w)];
title(titlestr)
colormap(jet)

print(f5,'-dpng','-r300','figure_8b_evollombscargle.png')

%%
% We can then compare the LS power spectrum with an FFT-based periodogram
% of the full time series which requires interpolation.
tint = min(data(:,1)):...
    (max(data(:,1))-min(data(:,1)))/...
    length(data(:,1)):max(data(:,1));
dataint = interp1(data(:,1),data(:,2),tint,...
    'pchip');

fs = 1/((max(data(:,1))-min(data(:,1)))/...
    length(data(:,1)));

[pfft,ffft] = ...
    periodogram(dataint,[],length(dataint),fs);

f6 = figure('Position',[50 ds(4)-1000 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
    'XLim',[0 0.15])
line(ffft,abs(pfft))
line([1/20 1/20],[0 max(abs(pfft))],...
    'Color','k')
line([1/40 1/40],[0 max(abs(pfft))],...
    'Color','k')
line([1/100 1/100],[0 max(abs(pfft))],...
    'Color','k')
text(1/20,max(abs(pfft))+0.1*max(abs(pfft)),...
    'Prec','HorizontalAlignment','center')
text(1/40,max(abs(pfft))+0.1*max(abs(pfft)),...
    'Obl','HorizontalAlignment','center')
text(1/100,max(abs(pfft))+0.1*max(abs(pfft)),...
    'Ecc','HorizontalAlignment','center')
xlabel('Frequency (1/kyr)')
ylabel('Power Spectral Density')
title('Periodogram')

print(f6,'-dpng','-r300','figure_5b_lombscargle.png')

%%
% We can use calculate the Thomson's multitaper power spectral density
% using the corresponding MATLAB function.
[ptmt,ftmt] = pmtm(dataint,1.25,length(dataint),fs,'centered');

f6 = figure('Position',[50 ds(4)-1000 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
    'XLim',[0 0.15],...
    'YLim',[0 50])
line(ftmt,ptmt)
line([1/20 1/20],[0 max(ptmt(:))],...
    'Color','k')
line([1/40 1/40],[0 max(ptmt(:))],...
    'Color','k')
line([1/100 1/100],[0 max(ptmt(:))],...
    'Color','k')
text(1/20,max(ptmt(:))+0.1*max(ptmt(:)),...
    'Prec','HorizontalAlignment','center')
text(1/40,max(ptmt(:))+0.1*max(ptmt(:)),...
    'Obl','HorizontalAlignment','center')
text(1/100,max(ptmt(:))+0.1*max(ptmt(:)),...
    'Ecc','HorizontalAlignment','center')
xlabel('Frequency (1/kyr)')
ylabel('Power Spectral Density')
title('Thomson s Multitaper Power Spectrum')

print(f6,'-dpng','-r300','figure_5b_multitaper.png')

%%
% We can compare the result with those of other methods of evolutionary
% power spectral analysis, such as the FFT-based spectrogram that also
% requires interpolation. We are using the function spectrogram from the
% Signal Processing Toolbox of MATLAB.
[ssp,fsp,tsp] = ...
    spectrogram(dataint,w,w-10,[],fs);

f8 = figure('Position',[50 ds(4)-1100 800 300],...
    'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
    'XLim',[0 1000],...
    'YLim',[0 0.15]); hold on
pcolor(tsp,fsp,(abs(ssp))), shading flat
line([min(data(:,1)) max(data(:,1))],...
    [1/20 1/20],...
    'Color','w')
line([min(data(:,1)) max(data(:,1))],...
    [1/40 1/40],...
    'Color','w')
line([min(data(:,1)) max(data(:,1))],...
    [1/100 1/100],...
    'Color','w')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/20,'Prec')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/40,'Obl')
text(max(data(:,1))+0.02*max(data(:,1)),...
    1/100,'Ecc')
title('Spectrogram')
xlabel('Time (kyr)')
ylabel('Frequency (1/kyr)')
set(gcf,'Colormap',jet)

print(f8,'-dpng','-r300','figure_8a_spectrogram.png')

%%
% Another method is wavelet power spectral analysis using cwt from the
% Wavelet Toolbox of MATLAB together with the default Morse mother wavelet
% that also requires interpolation.
[wt,fwt,coi] = cwt(dataint,fs);

f9 = figure('Position',[50 ds(4)-1200 800 300],...
'Color','w');
axes('Position',[0.1 0.15 0.8 0.7],...
'YLim',[0 0.15],...
'XGrid','On',...
'YGrid','On'); hold on
contour(tint,fwt,abs(wt),...
'LineStyle','none',...
'LineColor',[0 0 0],...
'Fill','on')
line(tint,coi,'Color','w',...
'LineStyle','--',...
'LineWidth',1)
line([min(data(:,1)) max(data(:,1))],...
[1/20 1/20],...
'Color','w')
line([min(data(:,1)) max(data(:,1))],...
[1/40 1/40],...
'Color','w')
line([min(data(:,1)) max(data(:,1))],...
[1/100 1/100],...
'Color','w')
text(max(data(:,1))+0.02*max(data(:,1)),...
1/20,'Prec')
text(max(data(:,1))+0.02*max(data(:,1)),...
1/40,'Obl')
text(max(data(:,1))+0.02*max(data(:,1)),...
1/100,'Ecc')
xlabel('Time (kyr)')
ylabel('Frequency (1/kyr)')
title('Wavelet Power Spectrum')
set(gcf,'Colormap',jet)

print(f9,'-dpng','-r300','figure_8c_wavelet.png')



